/* NeuQuant Neural-Net Quantization Algorithm Interface
* ----------------------------------------------------
*
* Copyright (c) 1994 Anthony Dekker
*
* NEUQUANT Neural-Net quantization algorithm by Anthony Dekker, 1994.
* See "Kohonen neural networks for optimal colour quantization"
* in "Network: Computation in Neural Systems" Vol. 5 (1994) pp 351-367.
* for a discussion of the algorithm.
* See also  http://members.ozemail.com.au/~dekker/NEUQUANT.HTML
*
* Any party obtaining a copy of these files from the author, directly or
* indirectly, is granted, free of charge, a full and unrestricted irrevocable,
* world-wide, paid up, royalty-free, nonexclusive right and license to deal
* in this software and documentation files (the "Software"), including without
* limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons who receive
* copies from any such party to do so, with the only requirement being
* that this copyright notice remain intact.
*/
 
//########################################################################
// Source code from members.ozemail.com.au/~dekker/NEUQUANT.H and
// members.ozemail.com.au/~dekker/NEUQUANT.C (retrieved on 2015-03-25)
// has been modified for integration by Socionext Embedded Software Austria (SESA)
//
// (C)  Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/BitmapConverter/NeuQuant.h>
#include <Candera/System/Diagnostics/Log.h>
#ifndef BUILD_FOR_PLUGIN_DLL
#include <Candera/Candera.h>
#endif

#ifdef BUILD_FOR_PLUGIN_DLL
#define NEU_QUANT_ALLOC_ARRAY(type, size)\
    new type[size]
#define NEU_QUANT_DELETE_ARRAY(pointer)\
    delete[] pointer
#else
#define NEU_QUANT_ALLOC_ARRAY(type, size)\
    FEATSTD_NEW_ARRAY(type, size)
#define NEU_QUANT_DELETE_ARRAY(pointer)\
    FEATSTD_DELETE_ARRAY(pointer)
#endif

namespace Candera
{
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);

    namespace ClutBitmapConverter  {
        NeuQuant::NeuQuant(Int32 colorCount) :
            m_netSize(colorCount), m_maxNetPos(0), m_netBiasShift(0), m_nCycles(0), m_intBiasShift(0), m_intBias(0),
            m_gammaShift(0), m_gamma(0), m_betaShift(0), m_beta(0), m_betaGamma(0),
            m_initRad(0), m_radiusBiasShift(0), m_radiusBias(0), m_initRadius(0), m_radiusDec(0),
            m_alphaBiasShift(0), m_initAlpha(0), m_alphaDec(0), m_radBiasShift(0), m_radBias(0),
            m_alphaRadBiasShift(0), m_alphaRadBias(0), m_thePicture(0), m_lengthCount(0), m_sampleFac(0),
            m_network(0), m_netIndex(0), m_bias(0), m_freq(0), m_radPower(0)
        {
            /* m_network Definitions
            ------------------- */

            m_maxNetPos = m_netSize - 1;
            m_netBiasShift = 4;       /* m_bias for color values */
            m_nCycles = 100;          /* no. of learning cycles */

            /* defs for m_freq and m_bias */
            m_intBiasShift = 16;          /* m_bias for fractions */
            m_intBias = static_cast<Int32>(static_cast<UInt32>(1) << m_intBiasShift);
            m_gammaShift = 10;            /* m_gamma = 1024 */
            m_gamma = static_cast<Int32>(static_cast<UInt32>(1) << m_gammaShift);
            m_betaShift = 10;
            m_beta = static_cast<Int32>(static_cast<UInt32>(m_intBias) >> m_betaShift);/* m_beta = 1/1024 */
            m_betaGamma = static_cast<Int32>(static_cast<UInt32>(m_intBias) << (m_gammaShift - m_betaShift));

            /* defs for decreasing radius factor */
            m_initRad = static_cast<Int32>(static_cast<UInt32>(m_netSize) >> 3);         /* for 256 cols, radius starts */
            m_radiusBiasShift = 6;            /* at 32.0 biased by 6 bits */
            m_radiusBias = static_cast<Int32>(static_cast<UInt32>(1) << m_radiusBiasShift);
            m_initRadius = m_initRad*m_radiusBias;    /* and decreases by a */
            m_radiusDec = 30;                     /* factor of 1/30 each cycle */

            /* defs for decreasing alpha factor */
            m_alphaBiasShift = 10;                /* alpha starts at 1.0 */
            m_initAlpha = static_cast<Int32>(static_cast<UInt32>(1) << m_alphaBiasShift);

            /* m_radBias and m_alphaRadBias used for m_radPower calculation */
            m_radBiasShift = 8;
            m_radBias = static_cast<Int32>(static_cast<UInt32>(1) << m_radBiasShift);
            m_alphaRadBiasShift = m_alphaBiasShift + m_radBiasShift;
            m_alphaRadBias = static_cast<Int32>(static_cast<UInt32>(1) << m_alphaRadBiasShift);


            m_network = NEU_QUANT_ALLOC_ARRAY(Pixel, m_netSize);       /* the m_network itself */

            m_netIndex = NEU_QUANT_ALLOC_ARRAY(Int32, 256);            /* for m_network lookup - really 256 */

            m_bias = NEU_QUANT_ALLOC_ARRAY(Int32, m_netSize);            /* m_bias and m_freq arrays for learning */
            m_freq = NEU_QUANT_ALLOC_ARRAY(Int32, m_netSize);
            m_radPower = NEU_QUANT_ALLOC_ARRAY(Int32, m_initRad);        /* m_radPower for precomputation */
        }

        NeuQuant::~NeuQuant()
        {
            NEU_QUANT_DELETE_ARRAY(m_network);
            NEU_QUANT_DELETE_ARRAY(m_netIndex);
            NEU_QUANT_DELETE_ARRAY(m_bias);
            NEU_QUANT_DELETE_ARRAY(m_freq);
            NEU_QUANT_DELETE_ARRAY(m_radPower);
            m_bias = 0;
            m_freq = 0;
            m_netIndex = 0;
            m_network = 0;
            m_radPower = 0;
            m_thePicture = 0;
        }

        /* Initialise network in range (0,0,0) to (255,255,255) and set parameters
        ----------------------------------------------------------------------- */

        void NeuQuant::InitNet(UInt8* thepic, Int32 len, Int32 sample)
        {
            Int32 i;
            Int32* p;

            m_thePicture = thepic;
            m_lengthCount = len;
            m_sampleFac = sample;

            for (i = 0; i < m_netSize; i++) {
                p = m_network[i].item;
                Int32 value = static_cast<Int32>((static_cast<UInt32>(i) << (m_netBiasShift + 8)) / m_netSize);
                p[0] = value;
                p[1] = value;
                p[2] = value;
                m_freq[i] = m_intBias / m_netSize;    /* 1/m_netSize */
                m_bias[i] = 0;
            }
        }


        /* Unbias m_network to give byte values 0..255 and record position i to prepare for sort
        ----------------------------------------------------------------------------------- */

        void NeuQuant::UnbiasNet()
        {
            Int32 i;
            Int32 j;
            Int32 temp;

            for (i = 0; i < m_netSize; i++) {
                for (j = 0; j<3; j++) {
                    /* OLD CODE: m_network[i][j] >>= m_netBiasShift; */
                    /* Fix based on bug report by Juergen Weigert jw@suse.de */
                    FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: >>): changing to unsigned type might change the end result of the expression")
                    temp = (m_network[i].item[j] + static_cast<Int32>(static_cast<UInt32>(1) << (m_netBiasShift - 1))) >> m_netBiasShift;
                    if (temp > 255) {
                        temp = 255;
                        m_network[i].item[j] = temp;
                    }
                }
                m_network[i].item[3] = i;       /* record color no */
            }
        }


        /* Output color map
        ----------------- */

        void NeuQuant::WriteColorMap(UInt8* clut) const
        {

            for (Int32 i = 0; i < m_netSize; i++) {
                *clut = 0xff;//a
                clut++;
                for (Int32 j = 0; j < 3; j++) {
                    *clut = UInt8(m_network[i].item[j]);
                    clut++;
                }
            }
        }

        Int32 NeuQuant::Contest(Int32 b, Int32 g, Int32 r)
        {
            /* finds closest neuron (min dist) and updates m_freq */
            /* finds best neuron (min dist-m_bias) and returns position */
            /* for frequently chosen neurons, m_freq[i] is high and m_bias[i] is negative */
            /* m_bias[i] = m_gamma*((1/m_netSize)-m_freq[i]) */

            Int32 i;
            Int32 dist;
            Int32 a;
            Int32 biasdist;
            Int32 betafreq;
            Int32 bestpos;
            Int32 bestbiaspos;
            Int32 bestd;
            Int32 bestbiasd;
            Int32* p;
            Int32* f;
            Int32* n;

            bestd = static_cast<Int32>(~((static_cast<UInt32>(1) << 31)));
            bestbiasd = bestd;
            bestpos = -1;
            bestbiaspos = bestpos;
            p = m_bias;
            f = m_freq;

            for (i = 0; i < m_netSize; i++) {
                n = m_network[i].item;
                dist = n[0] - b;
                if (dist < 0) {
                    dist = -dist;
                }
                a = n[1] - g;
                if (a < 0) {
                    a = -a;
                }
                dist += a;
                a = n[2] - r;   if (a < 0) {
                    a = -a;
                }
                dist += a;
                if (dist < bestd) {
                    bestd = dist; bestpos = i;
                }
                FEATSTD_LINT_CURRENT_SCOPE(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: >>): changing to unsigned type might change the end result of the expression")
                biasdist = dist - ((*p) >> (m_intBiasShift - m_netBiasShift));
                if (biasdist < bestbiasd) {
                    bestbiasd = biasdist;
                    bestbiaspos = i;
                }
                betafreq = (*f >> m_betaShift);
                *f++ -= betafreq;
                *p++ += (betafreq << m_gammaShift);
            }
            m_freq[bestpos] += m_beta;
            m_bias[bestpos] -= m_betaGamma;
            return(bestbiaspos);
        }


        /* Move neuron i towards biased (b,g,r) by factor alpha
        ---------------------------------------------------- */

        void NeuQuant::AlterSingle(Int32 alpha, Int32 i, Int32 b, Int32 g, Int32 r)
        {
            Int32* n;

            n = m_network[i].item; /* alter hit neuron */
            *n -= (alpha*(*n - b)) / m_initAlpha;
            n++;
            *n -= (alpha*(*n - g)) / m_initAlpha;
            n++;
            *n -= (alpha*(*n - r)) / m_initAlpha;
        }


        /* Move adjacent neurons by precomputed alpha*(1-((i-j)^2/[r]^2)) in m_radPower[|i-j|]
        --------------------------------------------------------------------------------- */

        void NeuQuant::AlterNeighbours(Int32 rad, Int32 i, Int32  b, Int32  g, Int32  r)
        {
            Int32 j;
            Int32 k;
            Int32 lo;
            Int32 hi;
            Int32 a;
            Int32 *p;
            Int32 *q;

            lo = i - rad;
            if (lo < -1) {
                lo = -1;
            }
            hi = i + rad;
            if (hi  >m_netSize) {
                hi = m_netSize;
            }

            j = i + 1;
            k = i - 1;
            q = m_radPower;
            while ((j<hi) || (k>lo)) {
                a = (*(++q));
                if (j<hi) {
                    p = m_network[j].item;
                    *p -= (a*(*p - b)) / m_alphaRadBias;
                    p++;
                    *p -= (a*(*p - g)) / m_alphaRadBias;
                    p++;
                    *p -= (a*(*p - r)) / m_alphaRadBias;
                    j++;
                }
                if (k>lo) {
                    p = m_network[k].item;
                    *p -= (a*(*p - b)) / m_alphaRadBias;
                    p++;
                    *p -= (a*(*p - g)) / m_alphaRadBias;
                    p++;
                    *p -= (a*(*p - r)) / m_alphaRadBias;
                    k--;
                }
            }
        }


        /* Main Learning Loop
        ------------------ */

        void NeuQuant::Learn()
        {
            Int32 i;
            Int32 j;
            Int32 b;
            Int32 g;
            Int32 r;
            Int32 radius;
            Int32 rad;
            Int32 alpha;
            Int32 step;
            Int32 delta;
            Int32 samplepixels = 0;
            UInt8 *p;
            UInt8 *lim;

            m_alphaDec = 30 + ((m_sampleFac - 1) / 3);
            p = m_thePicture;
            lim = m_thePicture + m_lengthCount;
            if (0 != m_sampleFac) {
                samplepixels = m_lengthCount / (3 * m_sampleFac);
            }
            delta = samplepixels / m_nCycles;
            alpha = m_initAlpha;
            radius = m_initRadius;
            FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: >>): changing to unsigned type might change the end result of the expression")
            rad = radius >> m_radiusBiasShift;
            if (rad <= 1) {
                rad = 0;
            }
            for (i = 0; i < rad; i++) {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(795, "For loop above ensures variable 'rad' can never be 0 in the line below.");
                m_radPower[i] = alpha*(((rad*rad - i*i)*m_radBias) / (rad*rad));
            }

            FEATSTD_LOG_DEBUG("beginning 1D learning: initial radius=%d", rad);

            if ((m_lengthCount%c_prime1) != 0) {
                step = 3 * c_prime1;
            }
            else {
                if ((m_lengthCount%c_prime2) != 0) {
                    step = 3 * c_prime2;
                }
                else {
                    if ((m_lengthCount%c_prime3) != 0) {
                        step = 3 * c_prime3;
                    }
                    else {
                        step = 3 * c_prime4;
                    }
                }
            }

            i = 0;
            while (i < samplepixels) {
                b = static_cast<Int32>(static_cast<UInt8>(p[0] << static_cast<UInt8>(m_netBiasShift))); //m_netBiasShift = 4
                g = static_cast<Int32>(static_cast<UInt8>(p[1] << static_cast<UInt8>(m_netBiasShift)));
                r = static_cast<Int32>(static_cast<UInt8>(p[2] << static_cast<UInt8>(m_netBiasShift)));
                j = Contest(b, g, r);

                AlterSingle(alpha, j, b, g, r);
                if (rad != 0) {
                    AlterNeighbours(rad, j, b, g, r);   /* alter neighbours */
                }

                p += step;
                FEATSTD_LINT_NEXT_EXPRESSION(946, "Relational or subtract operator applied to pointers [MISRA C++ Rule 5-0-15], [MISRA C++ Rule 5-0-17], [MISRA C++ Rule 5-0-18]: changing this would affect the algorithm ")
                if (p >= lim) {
                    p -= m_lengthCount;
                }

                i++;
                if (0 != delta) {
                    if (i % delta == 0) {
                        alpha -= alpha / m_alphaDec;
                        radius -= radius / m_radiusDec;
                        FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: >>): changing to unsigned type might change the end result of the expression")
                        rad = radius >> m_radiusBiasShift;
                        if (rad <= 1) {
                            rad = 0;
                        }

                        for (j = 0; j < rad; j++) {
                            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(795, "For loop above ensures variable 'rad' can never be 0 in the line below.");
                            m_radPower[j] = alpha*(((rad*rad - j*j)*m_radBias) / (rad*rad));
                        }
                    }
                }
            }
            FEATSTD_LOG_DEBUG("finished 1D learning: final alpha=%f !", static_cast<Float>(alpha / m_initAlpha));
        }
    }
}
